(* ::Package:: *)

BeginPackage["PolySGTools`", {"ComputationalGeometry`"}]
(* Authors: 
I. Garc\'{i}a-Garc\'{i}a <ignacio.garcia@uca.es>, 
A. S\'{a}nchez-R.-Navarro <alfredo.sanchez@uca.es>, 
A. Vigneron-Tenorio <alberto.vigneron@uca.es>
*)
PolygonalSG::usage = 
  "PolyginalSG[p], given p, a polygon rational 2D vertex set, if \
exists, it computes the minimal system of generators of the affine \
covex body semigroup associated to the convex hull of the polygon. \
The external program hilbert (a component of 4ti2) is needed. \
It must be assigned previously a value to the variable Solucionator \
similar to this example: \n \
Windows operating system: c:\\path\\zsolve -H \
Unix like operating system: /path/hilbert";
BelongToSG::usage = 
  "BelongToSG[p, v], given an integer pair p, and the set of vertex \
of a 2D rational v, it returns True if p belongs to the affine \
semigroup associated to the polygon, and False otherwise.
The external program hilbert (a component of 4ti2) is needed. \
It must be assigned previously a value to the variable Solucionator \
similar to this example: \n \
Windows operating system: c:\\path\\zsolve -H \
Unix like operating system: /path/hilbert";
SegmentSG::usage =
  "SegmentSG[a,b], given the extrems of an interval a<b in Q^2, \
computes a minimal system of generators of the semigroup generated by \
[a, b] cup N^2.
The external program hilbert (a component of 4ti2) is needed. \
It must be assigned previously a value to the variable Solucionator \
similar to this example: \n \
Windows operating system: c:\\path\\zsolve -H \
Unix like operating system: /path/hilbert";
PSGIsBuchsbaumQ::usage=
  "PSGIsBuchsbaumQ[p], given p, a polygon rational 2D vertex set, it returns True \
if the affine semigroup associated to the convex hull of the polygon is Buschsbaum.\
Otherwise, it returns False.
The external program hilbert (a component of 4ti2) is needed. \
It must be assigned previously a value to the variable Solucionator \
similar to this example: \n \
Windows operating system: c:\\path\\zsolve -H \
Unix like operating system: /path/hilbert";
Solucionator::usage=
  "This is a varible that must contain a string to the program \
hilbert (a component of 4ti2), which is needed in order to do some \
computations.";
CheckSolucionator::usage=
  "CheckSolucionator[], tests if the variable Solucionator is assigned to a \
valid value to perform the needed computations in the package.";


Begin["Private`"]


PolygonalSG[p_] := Module[
{i, pr = p, clws, lenclws, Ac, aux, pp, u, eq, ti, ToS,fname, s, equation, xmax, ymax, b},
   If[!ValueQ[Solucionator], Print["PolygonalSG::Solucionator variable not assigned."]; Abort[]];
   pr[[All,1]]=RationalToFrac[#]&/@ p[[All,1]];
   pr[[All,2]]=RationalToFrac[#]&/@ p[[All,2]];
   Needs["ComputationalGeometry`"];
   clws =ComputationalGeometry`ConvexHull[p,ComputationalGeometry`AllPoints -> False];lenclws = Length[clws];
   If[Length[clws] <= 2,
     Print["PolygonalSG::The number of points does not correspond to a rational polygon."];
     Return[];
   ];
   Ac = {};
   aux = 0*Range[1, lenclws];
   aux[[1]] = 1;
   pp = pr[[#]] & /@ clws;
   pp = Join[#, {1}] & /@ pp;
   For[i = 1, i <=  Length[clws], i++,
     u = pp[[1]]\[Cross]pp[[2]];
     u = u*(LCM @@ Denominator[u]);
     eq = Join[u, -Sign[u.pp[[3]]]*aux];
     aux = RotateRight[aux];
     pp = RotateRight[pp];
     AppendTo[Ac, eq];
   ];
   fname = FileNameJoin[{$TemporaryDirectory, "testfilepolsg"}];
   s = OpenWrite[fname<>".mat"];
   equation =ToString[Length[clws]]<> " " <> ToString[Length[Ac[[1]]]]<>"\n";
   WriteString[s,equation];
   Do[
      equation = "";
      Do[equation = equation <> " " <> ToString[Ac[[k, j]]],{j, 1, Length[Ac[[k]]]}];
      equation = equation <>"\n";
      WriteString[s, equation];,
   {k, 1, Length[Ac]}
   ];
   Close[s];
   Run[Solucionator<>" -q -p=64 "<>fname];
   If[!FileExistsQ[fname<>".hil"],Print["PolygonalSG:: Error: no solutions to get the generators."];Abort[];];
   ToS = Import[fname<>".hil","Table"];
   ToS = Drop[ToS,1];
   ToS = Union[ToS[[All, 1 ;; 2]]];
   pp = pp[[All, 1 ;; 2]];
   ToS = SistemaMGPol[pp, ToS];
   Return[ToS];
];


CheckSolucionator[] := Module[{fname, s, sols},
   If[!ValueQ[Solucionator], Print["CheckSolucionator::Solucionator variable not assigned."]; Abort[]];
   fname = FileNameJoin[{$TemporaryDirectory, "testsator"}];
   s = OpenWrite[fname<>".mat"];
   WriteString[s, "2 3\n-2 1 0\n0 3 -2"]; 
   Close[s];
   Run[Solucionator<>" -q "<>fname];
   If[!FileExistsQ[fname<>".hil"],Print["CheckSolucionator:: something went wrong. Check path of var Solucionator."];Abort[];];
   sols = Import[fname<>".hil","Table"];
   sols = Drop[sols,1];
   Return[sols == {{1, 2, 3}}];
];


(* Auxiliary functions *)
ecuacionesPoligonal[listapuntos_] := 
  Append[Table[ecuacionPar[listapuntos[[i]], listapuntos[[i + 1]]], {i, 1, Length[listapuntos] - 1}], ecuacionPar[listapuntos[[Length[listapuntos]]], 
   listapuntos[[1]]]];
(* Line equations through points (a1, a2) and (b1, b2) *)
ecuacionPar[{a1_, a2_}, {b1_, b2_}] := Det[( {{x - a1, y - a2}, {b1 - a1, b2 - a2} } )];
(* Check if the point (a, b) belongs to the polygon expressed with ecuaciones *)
estaEnInteriorOFrontera[{a_, b_}, ecuaciones_] := todoNegativoOCero[ecuaciones /. {x -> a, y -> b}];
todoNegativoOCero[l_] := And @@ (# <= 0 & /@ l);
perteneceASemigrupo[{a_, b_}, ecuaciones_, distancia_] := Module[
{k = 1, parar = False},
   While[Norm[{a, b}/k] >= distancia \[And] ! parar,
      parar = estaEnInteriorOFrontera[{a, b}/k, ecuaciones];
      k++;
   ];
   parar
];
distAlOrigen[listapuntos_] := Min[ Table[Norm[listapuntos[[i]]], {i, Length[listapuntos]}] ];


(* Computes the minimal system of generators of an ACBD given by the vertices set and a system of generators.
Input: 
conjvertices, convex polygon vertices
sistGeneradores, ACBS system of generators associated to conjvertices polygon
Output:
a minimal system of generators for the ACBS
*)
SistemaMGPol[conjvertices_, sistGeneradores_] := Module[
{menorIgualProdCartesiano, esElemMinimalPoligono, sistG, ecuaciones, dist},
   menorIgualProdCartesiano[a_, b_] :=   And @@ Table[a[[i]] <= b[[i]], {i, Length[a]}];
   (*******)
   esElemMinimalPoligono[ecuaciones_, dist_, sistG_, elemento_] := Module[
   {lmenores, i, longitud, seguir = True, cad}, 
      lmenores = Select[sistG, (menorIgualProdCartesiano[#, elemento] \[And] (# != elemento)) &];
      longitud = Length[lmenores];
      i = 1;
      While[i <= longitud \[And] seguir,
        If[perteneceASemigrupo[elemento - lmenores[[i]], ecuaciones, dist], seguir = False];
        i++;
      ];
      Return[seguir];
   ];
   (*******)
   sistG = sistGeneradores;
   ecuaciones = ecuacionesPoligonal[conjvertices];
   dist = distAlOrigen[conjvertices];
   Select[sistG, (esElemMinimalPoligono[ecuaciones, dist, sistG, #]) &]
];


BelongToSG[p_, vertices_] := Module[
{vertpol, pos = ConvexHull[vertices]},
   vertpol = Table[vertices[[ pos[[i]] ]], {i, 1, Length[pos]}];
   perteneceASemigrupo[p, ecuacionesPoligonal[vertpol], distAlOrigen[vertpol]]
];


(* Auxiliary functions *)
(* It returns the Hilbert basis of the matrix Ac. *)
Solve4ti2[Ac_]:=Module[
{fname,s,equation,res},
   fname = FileNameJoin[{$TemporaryDirectory, "testfile4ti2"}];
   s = OpenWrite[fname<>".mat"];
   equation =ToString[Length[Ac]]<> " " <> ToString[Length[Ac[[1]]]]<>"\n";
   WriteString[s,equation];
   Do[
      equation = "";
      Do[equation = equation <> " " <> ToString[Ac[[k, j]]],{j, 1, Length[Ac[[k]]]}];
      equation = equation <>"\n";
      WriteString[s, equation];,
   {k, 1, Length[Ac]}
   ];
   Close[s];
   Run[Solucionator <> " -q -p=64 " <> fname];
   If[!FileExistsQ[fname<>".hil"],Print["Solve4ti2:: no solutions to get the generators."];Abort[];];
   res = Import[fname<>".hil","Table"];
   Return[Drop[res,1]];
];


(* Auxiliary functions *)
(* It converts an expected rational value x to a fraction *)
RationalToFrac[x_] := Module[{res},
   res = Rationalize[N[x]];
   If[x - res <= 0.0000001, Return[res];,
     Print["RationalToFrac:: input ", x," too different from resulting rational ", res];
     Abort[];
   ];
];


(* Auxiliary functions *)
(* It returns the point intersection between two segments seg1 and seg2. *)
LineIntersectionPoint[seg1_,seg2_]:=Module[
{divider=Det[{seg1[[1]]-seg1[[2]],seg2[[1]]-seg2[[2]]}]},
   If[divider == 0,{},(Det[seg1](seg2[[1]]-seg2[[2]])-Det[seg2](seg1[[1]]-seg1[[2]]))/divider]
];


(* Auxiliary functions *)
(* It checks the existence of a natural points inside an region, see the description in paper. *)
NaturalsIntCBiQ[P_, n_, V_, jP_, t_] := Module[
{NaturalsInTriangleQ, vT, h, Th, existnatpoi},
  NaturalsInTriangleQ[Tk_]:=Module[{x,y},
   Return[Length[FindInstance[
     Sign[(x-Tk[[1,1]])*(Tk[[2,2]]-Tk[[1,2]])-((y-Tk[[1,2]])*(Tk[[2,1]]-Tk[[1,1]]))] == 
     Sign[(Tk[[3,1]]-Tk[[1,1]])*(Tk[[2,2]]-Tk[[1,2]])-((Tk[[3,2]]-Tk[[1,2]])*(Tk[[2,1]]-Tk[[1,1]]))] && 
     Sign[(x-Tk[[2,1]])*(Tk[[3,2]]-Tk[[2,2]])-((y-Tk[[2,2]])*(Tk[[3,1]]-Tk[[2,1]]))] == 
     Sign[(Tk[[1,1]]-Tk[[2,1]])*(Tk[[3,2]]-Tk[[2,2]])-((Tk[[1,2]]-Tk[[2,2]])*(Tk[[3,1]]-Tk[[2,1]]))] && 
     Sign[(x-Tk[[3,1]])*(Tk[[1,2]]-Tk[[3,2]])-((y-Tk[[3,2]])*(Tk[[1,1]]-Tk[[3,1]]))] == 
     Sign[(Tk[[2,1]]-Tk[[3,1]])*(Tk[[1,2]]-Tk[[3,2]])-((Tk[[2,2]]-Tk[[3,2]])*(Tk[[1,1]]-Tk[[3,1]]))]&& 
     x>=0 && y>=0,{x,y},Integers]]>0];
  ];
  vT = {{0, 0}, P, V};
  h = 0;
  Th = # + jP*P &/@ vT;
  existnatpoi = NaturalsInTriangleQ[Th];
  While[h < t && ! existnatpoi,
   Th = # + jP*P &/@ Th;
   existnatpoi = existnatpoi || NaturalsInTriangleQ[Th];
   h = h + 1;
  ];
  Return[existnatpoi];
];


(* Auxiliary functions *)
(* It checks if p belongs to the line defined by points p1 and p2. *)
InLineQ[p1_,p2_,p_]:=((p2[[2]]-p1[[2]])*p[[1]]-(p2[[1]]-p1[[1]])*p[[2]]-((p2[[2]]-p1[[2]])*p1[[1]]-(p2[[1]]-p1[[1]])*p1[[2]]))==0;


(* Auxiliary functions *)
(* It checks if p belongs to the segment defined by points p1 and p2. *)
InSegmentQ[p1_,p2_,p_]:=(p1!=p2) && (Sign[(p-p1).(p-p2)]<0);


(* Auxiliary functions *)
(* It computes the slopes of the extremal rays of the pointed cone that 
encloses the polygon defined by the vertices verts. *)
RaysSlopes[verts_]:=Module[{x0, y0, slopes, tau1, tau2},
   x0=Select[Select[verts,#!={0,0}&], #[[1]]==0&];
   If[x0!={}, tau1=\[Infinity],
   slopes=Flatten[{#[[2]]/#[[1]]}&/@Complement[verts,x0]];
      tau1:=Simplify[Max[slopes]];
      If[! Element[tau1, Rationals], Print["Irrational slope."]; Abort[]];
   ];
   y0=Select[Select[verts,#!={0,0}&], #[[2]]==0&];
   If[y0!={}, tau2=0,
   slopes=Flatten[{#[[2]]/#[[1]]}&/@Complement[Select[verts,#!={0,0}&],Union[x0,y0]]];
      tau2:=Simplify[Min[slopes]];
      If[! Element[tau2, Rationals], Print["Irrational slope."]; Abort[]];
   ];

   Return[{tau1, tau2}];
];


(* Auxiliary functions *)
(* It returns the set of natural points inside of the triangle defined by p1, p2, and p3
and on the lines whose slopes are m1 and m2. *)
SqTri[p1_,p2_,p3_,m1_,m2_]:=Module[
{cdl,cur,sqpts,fndis},
   TriOri[q1_,q2_,q3_]:=Sign[(q1[[1]]-q3[[1]])*(q2[[2]]-q3[[2]])-(q1[[2]]-q3[[2]])*(q2[[1]]-q3[[1]])];
  
   cdl = {Floor[Min[p1[[1]],p2[[1]],p3[[1]]]], Floor[Min[p1[[2]],p2[[2]],p3[[2]]]]};
   cur = {Ceiling[Max[p1[[1]],p2[[1]],p3[[1]]]], Ceiling[Max[p1[[2]],p2[[2]],p3[[2]]]]};
   sqpts = Flatten[Table[{i,j},{i,cdl[[1]], cur[[1]],1}, {j,cdl[[2]],cur[[2]],1}],1];
   sqpts = Select[sqpts,#[[2]]>= 0&];
   fndis = Function[t, 
     If[TriOri[p1,p2,p3]>= 0,
        If[TriOri[p1,p2,t]>=0 && TriOri[p2,p3,t]>= 0&&TriOri[p3,p1,t]>= 0,True,False],
        If[TriOri[p1,p2,t]<0 && TriOri[p2,p3,t]< 0&&TriOri[p3,p1,t]< 0,True,False]] 
        && If[m1==\[Infinity],t[[1]]!=0,t[[2]]!=m1*t[[1]]] 
        && If[m2==\[Infinity],t[[1]]!=0,t[[2]]!=m2*t[[1]]
     ]
   ];
   Return[Select[sqpts,fndis]];
];


(* Auxiliary functions *)
(* Given a list of 2d rational points, it returns the convex hull associated in fractional format. *)
ConvexPolygonVerts[pointlist_]:=Module[
{pos=ComputationalGeometry`ConvexHull[pointlist,ComputationalGeometry`AllPoints -> False],pch},
   If[Length[pos]<=2,
     Print["ConvexPolygonVerts:: the number of points does not correspond to a rational polygon."];
     Return[{}];
   ];
   pch = pointlist[[#]] &/@ pos;
   pch = {RationalToFrac[#[[1]]],RationalToFrac[#[[2]]]} &/@ pch;
   Return[pch];
];


PSGIsBuchsbaumQ[p_] := Module[
{lenpv, tau, n1, n2, iP1, iP2, P1, P2, 
j, inext, iprev, V1, V2, dists, t0, PSGgtau1, PSGgtau2,
B1, t1, B2, t2, n1p, n2p, nu1, nu2, Q, V, Vp, eqPolygon, dist0,
locBelongToSG, BelongToSGo, oPRayMinGen,
verts, polygen, Ups1, Ups2, Ups1ninoP, Ups2ninoP, Thpoints, Upsp, 
iCsetminusioP, j1, j2, isBuchsbaum},
(*******)
(**)
locBelongToSG[pto_]:=perteneceASemigrupo[pto, eqPolygon, dist0];
(* It checks if a point belongs to the set of points such that, if we add them with each element of g, 
the result belongs to the set generated with <g>.
Input:
pto, a point 
v, convex polygon vertices
g, ACBS system of generators associated to v
Output:
True o False, in case that p belongs or not to the set
*)
BelongToSGo[pto_]:=If[locBelongToSG[pto], True,
                     And @@ (locBelongToSG[pto+polygen[[#]]] &/@ Range[Length[polygen]])];

(* It computes the minimal generator on the ray with slope tau of the set whose elements are such 
that, if we add them with each element of polygen, the result belongs to the set generated 
with <polygen>
Input:
tau, slope of the ray
verts, polygon vertices
polygen, polygonal semigroup generators
*)
oPRayMinGen[tau_,pg_] := Module[{i, min},
   If[tau == \[Infinity],
     min = Min[polygen[[All, 2]]];
     For[i = 1, i < min, i++, 
       If[BelongToSGo[{0, i}], Return[{0, i}];];
     ];,
     min = Min[polygen[[All, 1]]];
     For[i = 1, i < min, i++, 
       If[BelongToSGo[{i, tau*i}], Return[{min, tau*min}];];
     ];
     Return[{min, tau*min}];
   ];
];
(*******)
   If[Length[p]==0, Print["Nothing to do with an empty input."]; Return[];];
   
   If[MemberQ[{0, 0},p], Return[True]];
   verts = ConvexPolygonVerts[p];
   lenpv = Length[verts];
   If[lenpv < 3, Print["The polygon must have 3 or more vertices."]; Abort[];];

   eqPolygon = ecuacionesPoligonal[verts];
   dist0 = distAlOrigen[verts];

   polygen = PolygonalSG[verts];
   tau=RaysSlopes[verts];
   (* Cone rays generators *)
   If[tau[[1]]==\[Infinity],n1={0,1},tau[[1]]=Simplify[tau[[1]]]; n1={Denominator[tau[[1]]],Numerator[tau[[1]]]}];
   If[tau[[2]]==0,n2={1,0},tau[[1]]=Simplify[tau[[1]]]; n2={Denominator[tau[[2]]],Numerator[tau[[2]]]}];
   If[tau[[1]]== \[Infinity],
     iP1=Select[Range[Length[verts]],verts[[#]][[1]]==0&],
     iP1=Select[Range[Length[verts]],verts[[#]][[2]]==verts[[#]][[1]]*tau[[1]]&];
   ];
   If[tau[[2]]==0,
     iP2=Select[Range[Length[verts]],verts[[#]][[2]]==0&],
     iP2=Select[Range[Length[verts]],verts[[#]][[2]]==verts[[#]][[1]]*tau[[2]]&];
   ];
   dists = Norm[#] & /@ verts;
   If[Length[iP1] > 1 ,
      iP1 = Sort[iP1, dists[[#1]] <= dists[[#2]] &];
      (* 1st segment where two of them intersect *)
      t0 =.;
      j1 = Minimize[{t0, (t0 + 1)*verts[[iP1[[1]], 1]] <= t0*verts[[iP1[[Length[iP1]]], 1]] && 
                  (t0 + 1)*verts[[iP1[[1]], 2]] <= t0*verts[[iP1[[Length[iP1]]], 2]] && t0 >= 0 && 
                  t0 \[Element] Integers}, {t0}][[1]];
   ];
   If[ Length[iP2] > 1,
      iP2 = Sort[iP2, dists[[#1]] <= dists[[#2]] &];
      t0 =.;
      j2 = Minimize[{t0, (t0 + 1)*verts[[iP2[[1]], 1]] <=  t0*verts[[iP2[[Length[iP2]]], 1]] && 
                  (t0 + 1)*verts[[iP2[[1]], 2]] <= t0*verts[[iP2[[Length[iP2]]], 2]] && t0 >= 0 && 
                  t0 \[Element] Integers}, {t0}][[1]];
   ];

   If[tau[[1]] == \[Infinity],
      PSGgtau1 = Select[polygen, #[[1]] == 0 &];,
      PSGgtau1 = Select[polygen, #[[1]]*tau[[1]] == #[[2]] &];
   ];
   P1 = Simplify[verts[[iP1[[1]]]]];

   If[tau[[2]] == 0,
      PSGgtau2 = Select[polygen, #[[2]] == 0 &];,
      PSGgtau2 = Select[polygen, #[[1]]*tau[[2]] == #[[2]] &];
   ];
   P2 = Simplify[verts[[iP2[[1]]]]];

   (* simple case *)
   If[Length[iP1] > 1  &&  Length[iP2] > 1,
      j = Max[j1, j2];
      Thpoints = SqTri[{0, 0}, j*P1, j*P2, tau[[1]], tau[[2]]];
      iCsetminusioP = Select[Thpoints,!BelongToSGo[#]&];
      Return[iCsetminusioP=={} && BelongToSGo[n1] && BelongToSGo[n2]];
   ];
   (* other situations *)
   If[Length[iP1] == 1,
      iprev = iP1[[1]] + 1; If[iprev > lenpv, iprev = 1;];
      inext = iP1[[1]] - 1; If[inext < 1, inext = lenpv;];

      V1 = LineIntersectionPoint[{P1, verts[[inext]]}, 2*{verts[[iprev]], P1}] - P1;
      (* Find j1, such that j1\overline{P1 P{1-1}}\cap j1\overline{P1 P{lenpv}}\n \eq \emptyset *)
      j =.;
      j1 = Minimize[j, InSegmentQ[(j + 1)*verts[[iprev]], (j + 1)*P1, j*P1 + V1] && 
                  InSegmentQ[j*verts[[inext]], j*P1, j*P1 + V1] && j >= 0, {j}, Integers][[1]];

      t1 = If[P1[[1]] == 0, 
            If[P1[[2]] == 0, 
               Print["PSGIsBuchsbaumQ:: {0,0} can not be a vertex.\n"]; Abort[];, 
               PSGgtau1[[1, 2]]/P1[[2]]
            ], 
            PSGgtau1[[1, 1]]/P1[[1]]
      ];
      If[! Element[t1, Integers], Print["PSGIsBuchsbaumQ:: Oppps, something is wrong with t1."]; Abort[]];
      
      If[ P1[[1]] != 0 && P1[[2]] != 0,
         If[t1 != PSGgtau1[[1, 2]]/P1[[2]],
            Print["PSGIsBuchsbaumQ:: Oppps, something is wrong with t1."]; Abort[]
         ];
      ];

      (* Check for integer points in interior (C)\cap B_ 1 *)
      B1 = NaturalsIntCBiQ[P1, PSGgtau1[[1]], V1, j1, t1];
      (* is int (C)\cap Ups1 \ oP = {}? *)
      nu1 = {V1 + n1, V1}; (* parallel to tau1 through V1 *)
      Q = LineIntersectionPoint[nu1, {n2, 2*n2}];
      Block[{topy, miny, iy, topx, minx, ix, m},
         vertUps1 = {{0, 0}, j1*P1, V1 + j1*P1, Q};
         topy = Floor[Max[vertUps1[[2, 2]], vertUps1[[3, 2]]]];
         Ups1 = {}; (* inside Ups1 parallelogram *)
         m = (vertUps1[[3, 1]] - vertUps1[[2, 1]]); (* slope of line j1P1 to V1 *)
         If[m != 0, m = (vertUps1[[3, 2]] - vertUps1[[2, 2]])/m, m = \[Infinity]];
         For[iy = 1, iy <= topy, iy++,
            If[vertUps1[[2, 2]] >= vertUps1[[3, 2]],
               minx = Ceiling[iy/tau[[1]]];
               topx = Floor[If[iy<=Q[[2]], 
                               If[tau[[2]]==0, Q[[1]], iy/tau[[2]]], 
                               If[iy <= vertUps1[[3, 2]],
                                   (iy - nu1[[1, 2]])/tau[[1]] + nu1[[1, 1]], 
                                   If[m == \[Infinity], vertUps1[[3, 1]], (iy - vertUps1[[2, 2]])/m + vertUps1[[2, 1]]]
                               ]
                            ]
                           ];,
               minx = Ceiling[If[iy <= vertUps1[[2, 2]], iy/tau[[1]], (iy - vertUps1[[2, 2]])/m + vertUps1[[2, 1]]]];
               topx = Floor[If[iy<=Q[[2]], If[tau[[2]]==0, Q[[1]], iy/tau[[2]]],(iy - nu1[[1, 2]])/tau[[1]] + nu1[[1, 1]]]];
            ];

            If[Length[iP2] > 1 && iy < vertUps1[[4, 2]], topx = Floor[iy/tau[[2]]];];

            For[ix = minx, ix <= topx, ix++,
               If[! InLineQ[{0, 0}, P1, {ix, iy}] && ! InLineQ[{0, 0}, P2, {ix, iy}],
                  Ups1 = Ups1~Join~{{ix, iy}};
               ];
            ];
         ];
      ];

      Ups1ninoP = Select[Ups1, !BelongToSGo[#] &];
      If[Length[iP2] > 1,
         j = Max[j1, j2];
         Thpoints = SqTri[Q, j*P1 + V1, j*P2, tau[[1]], tau[[2]]];
         iCsetminusioP = Select[Thpoints, ! BelongToSGo[#] &];
         If[iCsetminusioP == {} && Ups1ninoP == {},
            isBuchsbaum = BelongToSGo[n2];,
            If [iCsetminusioP !=  {}, isBuchsbaum = False,
               (* Find oP\cap tau2 min generator *)
               ix = n2[[1]];
               While[ix <= Min[PSGgtau2[[All, 1]]],
                  If[! BelongToSGo[{ix, ix*tau[[2]]}], ix++;, Break[];];
               ];
               n2p = {ix, tau[[2]]*ix};
               (* Check Ups', knowing that Ups2={} *)
               Upsp = Ups1ninoP;
               (* Find one in Upsp *)
               While[Length[Upsp] > 0,
                  If[BelongToSGo[Upsp[[1]] + PSGgtau1[[1]]] && 
                     BelongToSGo[Upsp[[1]] + n2p],
                     Break[];,
                     Upsp = Delete[Upsp, 1];
                  ];
               ];
               If[Upsp != {}, isBuchsbaum = False;, isBuchsbaum = True;];
            ];
         ];
         Return[isBuchsbaum];
      ];
   ];

   If[Length[iP2] == 1,
      iprev = iP2[[1]] - 1; If[iprev < 1, iprev = lenpv;];
      inext = iP2[[1]] + 1; If[inext > lenpv, inext = 1;];

      V2 = LineIntersectionPoint[{P2, verts[[inext]]}, 2*{verts[[iprev]], P2}] - P2;
      j =.;
      j2 = Minimize[j, InSegmentQ[(j + 1)*verts[[iprev]], (j + 1)*P2, j*P2 + V2] && 
                  InSegmentQ[j*verts[[inext]], j*P2, j*P2 + V2] && j >= 0, {j}, Integers][[1]];
      t2 = If[P2[[1]] == 0, 
            If[P2[[2]] == 0, 
               Print["PSGIsBuchsbaumQ:: {0,0} can not be a vertex.\n"]; Abort[];, 
               PSGgtau2[[1, 2]]/P2[[2]]
            ], 
            PSGgtau2[[1, 1]]/P2[[1]]
      ];

      If[! Element[t2, Integers], Print["PSGIsBuchsbaumQ:: Oppps, something is wrong with t2."]; Abort[]];
      If[P2[[1]] != 0 && P2[[2]] != 0, 
         If[t2 != PSGgtau2[[1, 2]]/P2[[2]], 
            Print["PSGIsBuchsbaumQ:: Oppps, something is wrong with t2."]; Abort[];
         ];
      ];

      B2 = NaturalsIntCBiQ[P2, PSGgtau2[[1]], V2, j2, t2];
      nu2 = {V2 + n2, V2};

      Q = LineIntersectionPoint[{n1, 2*n1}, nu2];
      Block[{topy, miny, iy, topx, minx, ix, m},
         vertUps2 = {{0, 0}, j2*P2, V2 + j2*P2, Q };

         topx = Floor[Max[vertUps2[[2, 1]], vertUps2[[3, 1]]]];
         Ups2 = {};
         m = (vertUps2[[3, 1]] - vertUps2[[2, 1]]);
         If[m != 0, m = (vertUps2[[3, 2]] - vertUps2[[2, 2]])/m, m = \[Infinity]];
         For[ix = 1, ix <= topx, ix++,
            If[vertUps2[[3,1]]>=vertUps2[[2,1]],
               miny=Ceiling[If[ix <= vertUps2[[2,1]], 
                               tau[[2]]*ix, 
                               If[m == \[Infinity], vertUps2[[3,2]],m*(ix - vertUps2[[2, 1]]) + vertUps2[[2, 2]]]]];
               topy=Floor[If[ix <= Q[[1]], 
                             If[tau[[1]] == \[Infinity], Q[[2]], tau[[1]]*ix], tau[[2]]*(ix - nu2[[1, 1]]) + nu2[[1, 2]]]],
               miny=Ceiling[tau[[2]]*ix];
               topy=Floor[If[ix <= Q[[1]], If[tau[[1]] == \[Infinity], Q[[2]], tau[[1]]*ix], 
                             If[ix <= vertUps2[[3,1]],
                                tau[[2]]*(ix - nu2[[1, 1]]) + nu2[[1, 2]], m*(ix - vertUps2[[2, 1]]) + vertUps2[[2, 2]]]]]
            ];
            If[Length[iP1] > 1 && ix < vertUps2[[4, 1]], topy = Floor[tau[[1]]*ix]];

            For[iy = miny, iy <= topy, iy++,
               If[! InLineQ[{0, 0}, P1, {ix, iy}] && ! InLineQ[{0, 0}, P2, {ix, iy}],
                  Ups2 = Ups2~Join~{{ix, iy}};
               ];
            ];
         ];
      ];
      Ups2ninoP = Select[Ups2, ! BelongToSGo[#] &];
      If[Length[iP1] > 1,
         j = Max[j1, j2];
         Thpoints = SqTri[Q, j*P2 + V2, j*P1, tau[[1]], tau[[2]]];
         iCsetminusioP = Select[Thpoints, ! BelongToSGo[#] &];
         If[iCsetminusioP == {} && Ups2ninoP == {},
            isBuchsbaum = BelongToSGo[n1];,
            If[iCsetminusioP !=  {}, isBuchsbaum = False,
               (* Find oP\cap tau1 min generator *)
               ix = n1[[1]];
               While[ix <= Min[PSGgtau1[[All, 1]]],
                  If[! BelongToSGo[{ix, ix*tau[[1]]}], ix++;, Break[];];
               ];
               n1p = {ix, tau[[1]]*ix};
               (* Check Ups', knowing that Ups1={} *)
               Upsp = Ups2ninoP;
               (* Find one in Upsp *)
               While[Length[Upsp] > 0,
                  If[BelongToSGo[Upsp[[1]] + PSGgtau2[[1]]] && 
                     BelongToSGo[Upsp[[1]] + n1p],
                     Break[];,
                     Upsp = Delete[Upsp, 1];
                  ];
               ];
   
               If[Upsp != {}, isBuchsbaum = False;, isBuchsbaum = True;];
            ];
         ];
         Return[isBuchsbaum];
      ];
   ];
         (****)
   If[Length[iP1] == 1 && Length[iP2] == 1,
      Q = LineIntersectionPoint[nu1, nu2];
      j = Max[j1, j2];
      V = j*P1 + V1; If[Q == V, V = V + P1];
      Vp = j*P2 + V2; If[Q == Vp, Vp = Vp + P2];
      iCsetminusioP =  Select[SqTri[Q, V, Vp, tau[[1]], tau[[2]]], ! BelongToSGo[#] &];
      If[! B1 && Ups1ninoP == {} && ! B2 && Ups2ninoP == {} && iCsetminusioP == {},
         isBuchsbaum = True;,
         
         Upsp = Union[Ups1ninoP, Ups2ninoP];
         While[Length[Upsp] > 0,
            If[BelongToSGo[Upsp[[1]] + PSGgtau1[[1]]] && 
               BelongToSGo[Upsp[[1]] + PSGgtau2[[1]]],
               Break[];
            ];
            Upsp = Delete[Upsp, 1];
         ];
         If[Upsp == {} && iCsetminusioP == {}, isBuchsbaum = True;, isBuchsbaum = False;];
      ];
      Return[isBuchsbaum];
   ];
];


End[]
EndPackage[];



